function th = threshold_update(ydata, enable_hist_plotting, rid_read, peak_amps_StartPoint, peak_positions_StartPoint)
% THRESHOLD_UPDATE - this function outputs the optimal threshold for photon discrimination based on the poissonian 
%                    fit to the photon counting data; if requested plots a photon histogram of the input data                    
%   INPUTS:
%       ydata - 1D array or 2D matrix containing the photon counts (collected within time bin) from the experiment ;
%               
%       enable_hist_plotting - bool (true - plot histogram of photon counts)                      
%       rid_read - data file rid number, relevant for plotting the histogram 
% 
%   OPTIONAL INPUTS:
%       peak_amps_StartPoint - 1D array: [amp_peak1, amp_peak2], starting amps for fitting poissonians 
%       peak_positions_StartPoint - 1D array: [count_peak1, count_peak2], starting counts for fitting poissonians
%       
%   OUTPUTS:
%       th - int, updated threshold value for the threshold obtained with the method defined in 'th_det_method'


% If not provided set the initial starting points for fitting poissonians to histograms 
if ~exist('peak_amps_StartPoint','var'), peak_amps_StartPoint=[1e4,1e4]; end

% Low thresholds enable - it is for low mean photon numbers, i.e. when there is negligible background
low_threshold_enable = 0 ;

% 1. Fit the double-Poissonian to photon counts and extract threshold 
[hist_counts, hist_edges] = histcounts(ydata,'BinWidth',1) ;

% Update starting positions for the peaks
if ~low_threshold_enable
%     start_pos1 = (max(hist_edges) + min(hist_edges)) /2 - 40 ;
%     start_pos2 = (max(hist_edges) + min(hist_edges)) /2 + 20 ;
    
    start_pos1 = 20 ;
    start_pos2 = max(hist_edges) - 20 ; 
    
else
    % Enabled low photon numbers -> use lower numbers for peak determination
    start_pos1 = 20 ;
    start_pos2 = 80 ;
end    



if ~exist('peak_positions_StartPoint','var') || isnan(peak_positions_StartPoint)
    peak_positions_StartPoint=[start_pos1,start_pos2] ; 
end 



% Keep searching Poissonian fits to the histogram of photon counts until reasonably spaced
while true 
    % Fitting double Poissonian to the photon counts
    fitfunPoissonSingle = @(a1,lambda1,x) a1*pdf('Poisson',x,lambda1) ;
    fitfunPoissonDouble = @(a1,a2,lambda1,lambda2,x) a1*pdf('Poisson',x,lambda1) + a2*pdf('Poisson',x,lambda2) ;
    ft_PoissonDouble = fittype(fitfunPoissonDouble,'independent','x','dependent','y');

    opts = fitoptions('Method','NonlinearLeastSquares');
    opts.Display = 'Off';
    opts.Lower = [0 0 hist_edges(1) hist_edges(1)];

    opts.StartPoint = [peak_amps_StartPoint(1) peak_amps_StartPoint(2) peak_positions_StartPoint(1) peak_positions_StartPoint(2)];

    x_range = hist_edges(1):hist_edges(end)-1 ;
    [fitresultPhoton_P1, ~] = fit(x_range',hist_counts',ft_PoissonDouble,opts) ;
    x_range_updated = hist_edges(1):hist_edges(end)-1 ;

    p1_name = sprintf('Mean: %.0f photons',fitresultPhoton_P1.lambda1 ) ;
    p2_name = sprintf('Mean: %.0f photons',fitresultPhoton_P1.lambda2 ) ;
    
    p1_data = fitfunPoissonSingle(fitresultPhoton_P1.a1,fitresultPhoton_P1.lambda1,x_range_updated) ; 
    p2_data = fitfunPoissonSingle(fitresultPhoton_P1.a2,fitresultPhoton_P1.lambda2,x_range_updated) ;
    
    % Checking positions and heights of the peaks 
    if false 
        if max(p1_data) < 7 || max(p2_data) < 7 % Check if height of the peak makes sense
            peak_positions_StartPoint(1) = peak_positions_StartPoint(1) + 10 ;
            peak_positions_StartPoint(2) = peak_positions_StartPoint(2) + 20 ;
        else  
            % Check if the poissonian fit peaks are making sense - recorrect positions with respect to the left peak 
            if abs(fitresultPhoton_P1.lambda2 - fitresultPhoton_P1.lambda1) > 60 || abs(fitresultPhoton_P1.lambda2 - fitresultPhoton_P1.lambda1) < 20
                % Define peak 1 as the left one 
                peak_positions_StartPoint(1) = min([peak_positions_StartPoint(1), peak_positions_StartPoint(2)]) ;
                % Shift the peak to the right 
                peak_positions_StartPoint(2) = peak_positions_StartPoint(1) + 50 ;
            else 
                break
            end
        end    
    else
        break % No checks performed
    end 

end

 

% Find crossing of the poissonians to determine optimal threshold (sometimes indices inverted)
try 
    differences = abs(p1_data - p2_data) ; 
    differences = (x_range_updated>fitresultPhoton_P1.lambda1).* differences ;
    differences = (x_range_updated<fitresultPhoton_P1.lambda2).* differences ;
    suggested_th = x_range_updated(find(differences == min(differences(differences>0)))) ;
catch
    differences = abs(p1_data - p2_data)  ; 
    differences = (x_range_updated<fitresultPhoton_P1.lambda1).* differences ;
    differences = (x_range_updated>fitresultPhoton_P1.lambda2).* differences ;
    try
        suggested_th = x_range_updated(find(differences == min(differences(differences>0)))) ; 
    catch
        suggested_th = 35 ; %skip problematic thresholds -- overwrite if required
    end
end

if enable_hist_plotting
    % Plotting the data as a histogram
    figure ;
    h1 = histogram(ydata,'BinWidth',1,'DisplayName','histogram data') ;
    xlabel('Number of photons')
    ylabel('Counts')
    legend()
    hold on

    p0 = plot(x_range_updated,fitfunPoissonDouble(fitresultPhoton_P1.a1,...
        fitresultPhoton_P1.a2,fitresultPhoton_P1.lambda1,...
        fitresultPhoton_P1.lambda2,x_range_updated), ...
        '-', 'linewidth', 2,'DisplayName','Double-Poisson Fit') ;
    
    p1 = plot(x_range_updated,p1_data,'--', 'linewidth', 1, 'DisplayName',p1_name, 'linewidth', 1.5) ;
    p2 = plot(x_range_updated,p2_data,'--', 'linewidth', 1,'DisplayName',p2_name, 'linewidth', 1.5) ;
    
    suggested_th_description = sprintf('Suggested threshold: %.0f',suggested_th ) ;
    xline(suggested_th, '--','DisplayName',suggested_th_description,'Linewidth',2) ;

    title_str = sprintf('Brights and darks - Double Poissonian Fit. File: %d', rid_read) ;
    title(title_str) ;
    
    box on;
    set(gca, 'FontSize', 14);
    
    hold off

end


% Overwrite threshold 
th = suggested_th ;

end
   
